/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

if (!window.Picasso.hasOwnProperty("Versions")) {
    window.Picasso.Versions = {};
}

/* Private scope */
(function () {

    /* Globals */
    var TABLE = new Picasso.Table("#versions_table",
        Picasso.Table.Flags.ROWS_SELECTABLE);
    var PAGE = 1;

    /**
     * Filter deleted files
     *
     * @param {Array}  files  Array of {@link Picasso.File}
     *
     * @returns {Array} Filtered list; might be empty
     **/

    var filterDeleted = function (files) {
        var filtered = files.filter(function (file) {
            return (file.is(Picasso.File.Flags.FOLDER|
                Picasso.File.Flags.DIRECTORY|Picasso.File.Flags.DELETED));
        });

        return filtered;
    };

    /**
     * Get selected file
     *
     * @param {jQuery|HTMLElement}  elem  Element to check
     *
     * @returns {*} Either found #{@link Picasso.File}; otherwise null
     **/

    var getFileFromElem = function (elem) {
        var file = null;
        var serial = $(elem).find("[data-serial]");

        if (0 < serial.length) {
            file = new Picasso.File(serial.data("serial"));
        }

        return file;
    };

    /**
     * Handle click on element name
     *
     * @param {Event}  e  Click event
     **/

    var handleClickName = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var file = new Picasso.File($(this).data("serial"));

        /* Check type */
        if (file.is(Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY)) {
            TABLE.deselectAll();

            Picasso.Versions.fetch(file, false,

                /* Success */
                function () {
                    var a = Picasso.Actionbar.addCrumb(
                        file.getName(), handleClickCrumb);

                    a.data("serial", file.serialize());
                }
            );
        }
    };

    /**
     * Handle click on bread bar
     **/

    var handleClickCrumb = function () {
        var file = $(this).data("serial");

        if (file) {
            file = new Picasso.File(file);
        }

        Picasso.Versions.fetch(file);

        /* Clear search field */
        $("#pica_action_search_input").val("");
    };

    /**
     * Handle click on purge button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickPurge = function (e) {
        e.preventDefault();
        e.stopPropagation();

        /* Get elements:
         * 1) All selected
         * 2) Current selected one
         * 2) All rows */
        var selectedFiles = TABLE.getSelectedRows();

        if (0 === selectedFiles.length) {
            var closest = getFileFromElem(this.closest("tr"));

            if (closest) {
                selectedFiles = [ closest ];
            }
        }

        if (0 === selectedFiles.length) {
            selectedFiles = TABLE.getRows();
        }

        if (0 === selectedFiles.length) {
            return;
        }

        selectedFiles = $.map(selectedFiles, function (val) {
            return getFileFromElem(val);
        });

        /* Show dialog */
        var what = (1 === selectedFiles.length ? selectedFiles[0].getName() :
            Picasso.Lang.get("string_selected_elements", selectedFiles.length));

        var confirmDialog = new Picasso.Dialog("#pica_confirmation_dialog",
            Picasso.Lang.get("dialog_title_purge"),
            Picasso.Lang.get("dialog_body_purge", what));

        confirmDialog.setOkHandler(function () {
            var syncCount = selectedFiles.length;

            /* Show waiting dialog */
            var waitingDialog = new Picasso.Dialog("#pica_waiting_dialog");

            waitingDialog.show();

            var hideDialog = function () {
                syncCount--;

                if (0 === syncCount) {
                    waitingDialog.hide();
                }
            };

            $.each(selectedFiles, function () {
                var file = this;

                file.purge(

                    /* Success */
                    function () {
                        Picasso.Helper.fadeOut($("#" + file.getElemId()),
                            function () {
                                this.remove();

                                TABLE.checkIfEmpty();
                            });

                        Picasso.Notification.show(Picasso.Lang.get(
                            "notification_item_purged", Picasso.Lang.get(
                                file.is(Picasso.File.Flags.FOLDER) ?
                                    "label_folders" : "label_files")), "info");

                        hideDialog();
                    },

                    /* Error */
                    function (e) {
                        Picasso.debugLog(e);

                        hideDialog();
                    }
                );
            });
        });

        confirmDialog.show();
    };

    /**
     * Handle click on restore button
     *
     * @param {Event}  e  Click event
     **/

    var handleClickRestore = function (e) {
        e.preventDefault();
        e.stopPropagation();

        var selected = TABLE.getSelectedRows();

        if (1 === selected.length) {
            var file = getFileFromElem(selected[0]);
            var dialog = new Picasso.Dialog.Restore(file);

            dialog.show();
        }
        else {
            selected = $.map(selected, function (val) {
                return getFileFromElem(val);
            });

            let notification = "notification_item_restored";
             $.each(selected, function () {
                    var file = this;
                    file.restore(file.getVersion()-1,

                            /* Success */
                            function () {
                                Picasso.Helper.fadeOut($("#" + file.getElemId()),
                                        function () {
                                            this.remove();

                                            TABLE.checkIfEmpty();
                                        });

                                Picasso.Notification.show(
                                        Picasso.Helper.capitalize(
                                                Picasso.Lang.get(notification, Picasso.Lang.get(
                                                        file.is(Picasso.File.Flags.FOLDER) ?
                                                                "label_folders" :
                                                                (file.is(Picasso.File.Flags.DIRECTORY) ?
                                                                        "label_directories" : "label_files")))), "info");
                                file.setVersion(file.getVersion()+1);
                            },

                            /* Error */
                            function (e) {
                                if (file.is(Picasso.File.Flags.FEDERATED)) {
                                    Picasso.Notification.show(
                                            Picasso.Lang.get("notification_federated_service_not_reachable"), "warning");
                                } else {
                                    Picasso.debugLog(e);
                                }
                            }
                    );
                });


           }
    };

    /**
     * Render given json object
     *
     * @param {Picasso.File}  file  A {@type Picasso.File}
     *
     * @returns {jQuery|HTMLElement} Rendered object
     **/

    var renderVersion = function (file) {
        var tr = $("<tr>", {
            id: file.getElemId(),
            "data-search-keys": [file.getName(),file.getOid()],
            draggable: false
        });

        /* 1. column: Icon */
        var td = $("<td>", { class: "text-center pica-icon-column" });

        var icon = file.getIcon(); ///< FIXME: Thumbnails have been deleted

        td.append(icon);
        tr.append(td);

        /* 2. column: Name */
        td = $("<td>", { class: "pica-overflow" });

        var a = $("<a>", {
            class: "pica-name",
            text: file.getName(),
            "data-serial": file.serialize()
        });

        a.click(handleClickName);

        td.append(a);
        tr.append(td);

        /* 3. column: Size/Permissions */
        td  = $("<td>");

        if (file.is(Picasso.File.Flags.FOLDER)) {
            td.append(Picasso.Lang.get("permission_you_" +
                Picasso.Permission.toString(file.getPermission()).toLowerCase()));
        } else {
            td.append(Picasso.Helper.formatSize(file.getSize()));
            td.attr("data-sort", file.getSize());
        }

        tr.append(td);

        /* 3. column: Versions */
        td  = $("<td>", { class: "hidden-xs" });

        if (!file.is(Picasso.File.Flags.FOLDER)) {
            td.append(file.getVersion());
        }else{
            td.append(Picasso.Helper.formatSize(file.getArchiveSize()));
            td.attr("data-sort", file.getArchiveSize());
        }

        tr.append(td);

        /* 4. column: Deleted at */
        td = $("<td>", {
            class: "hidden-sm hidden-xs",
        });

        if (file.is(Picasso.File.Flags.DELETED)) {
            /* Create tooltip */
            var span = $("<span>", {
                text: Picasso.Helper.formatAge(file.getModifiedDate()),
                title: Picasso.Helper.formatDate(file.getModifiedDate()),
                "data-tooltip": "",
                "data-placement": "bottom"
            });

            td.attr("data-sort", Date.parse(file.getModifiedDate()) || 0); ///< We need that for sorting
            td.append(span);

            span.tooltip({ container: "body" });
        }

        tr.append(td);

        return tr;
    };

    /**
     * Update table header for selection mode
     *
     * @param {Integer}  nselected    Number of selected rows
     * @param {Object}   selectedRow  Selected row object
     **/

    var handleOnSelection = function (nselected, selectedRow) {

        var cwd = Picasso.get("cwd");
        var lowestPermission = 0;
        if (cwd) {
            lowestPermission = cwd.getPermission();
        } else {
            lowestPermission = Picasso.Permission.OWNER;
        }

        if (1 <= nselected) {
            lowestPermission = Picasso.Permission.OWNER;

            $.each(TABLE.getSelectedRows(), function () {
                var selectedFile = getFileFromElem(this);

                if (selectedFile) {
                    var permission = selectedFile.getPermission();
                    if (permission < lowestPermission) {
                        lowestPermission = permission;
                    }
                }
            });
        }

        var visibilities = getVisibilities(lowestPermission);
        toggleVisibility(visibilities);


        var onlyRoot = $(".archive-visible-only-root");
        var onlyOther = $(".archive-visible-only-other");

        /* Toggle visibility for root */
        var cwd = Picasso.get("cwd");

        if (cwd) {
            onlyRoot.hide();
            onlyOther.show();
        } else {
            onlyRoot.show();
            onlyOther.hide();
        }
    };

    /**
     * Toggle element visibilities
     *
     * @param {Number}  visibilities  Element visibilities
     **/

    var toggleVisibility = function (visibilities) {
        $.each([
            "#versions_restore",
            "#versions_purge"
        ], function (idx) {
            var elems = $(this);

            if (0 < (visibilities & (1 << idx))) {
                elems.show();
            } else {
                elems.hide();
            }
        });
    };

    /**
     * Get visibilities for {@link Picasso.Permission}
     *
     * @param {Picasso.Permission}  lowestPermission      A #{@link Picasso.Permission}
     **/

    var getVisibilities = function (lowestPermission)
    {
        var visibilities = 0;
        if (lowestPermission === Picasso.Permission.OWNER || lowestPermission === Picasso.Permission.ADMIN ) {
            visibilities |= (
                    Picasso.Versions.Visibility.RESTORE |
                    Picasso.Versions.Visibility.PURGE
            );
        }else if(lowestPermission === Picasso.Permission.READ_WRITE){
            visibilities |= (
                    Picasso.Versions.Visibility.RESTORE
            );
        }
        return visibilities;
    };

    /**
     * Handle filter event
     *
     * @param {String}  filterString  Input string for filter
     * @param {Number}  nmatches      Number of visible filter matches
     **/

    var handleFilter = function (filterString, nmatches) {
        var cwd = Picasso.get("cwd");

        /* Update files list */
        var updateFiles = function (files) {
            var filtered = filterDeleted(files);

            TABLE.hideEmptyMessage();

            if (0 < files.length) {
                TABLE.update(filtered, renderVersion);
            } else {
                TABLE.showEmptyMessage();
            }
        };

        /* Limit search */
        if (0 === nmatches && 3 <= filterString.length) {
            if (cwd) {
                cwd.getFiles(updateFiles, null, null, null,
                    true, filterString);
            } else {
                Picasso.File.getAll(updateFiles, null, null, null,
                    true, filterString)
            }
        } else if (0 === nmatches && 0 < filterString.length) {
            TABLE.showEmptyMessage();
        } else {
            TABLE.hideEmptyMessage();
        }
    };

    /**
     * Fetch JSON data
     *
     * @param {Picasso.File}  file       A #{@link Picasso.File}
     * @param {Boolean}       append     Whether to append data
     * @param {Function}      onSuccess  Success handler
     * @param {Function}      onError    Error handler
     **/

    Picasso.Versions.fetch = function (file, append, onSuccess, onError) {
        Picasso.set("cwd", file);

        /* Reset page */
        if (true !== append) {
            PAGE = 1;
        }

        /* Update file list */
        var updateFiles = function (files, json) {
            var filtered = filterDeleted(files);

            /* Check for folder permission */
            if (json.hasOwnProperty("ResultSet") &&
                    json.ResultSet.hasOwnProperty("permission"))
            {
                Picasso.set("folderPermission", Picasso.Permission.fromString(
                        json.ResultSet.permission));
            } else {
                Picasso.unset("folderPermission");
            }

            const lessThanAdmin = (element) => element.getPermission() < Picasso.Permission.ADMIN;

            if (files.some(lessThanAdmin)) {
                //Hide pure all if one or more folders are read/write
                $("#versions_purge_all").hide();
            }

            /* Call renderer before infinite scroll */
            if (true === append) {
                TABLE.append(filtered, renderVersion);
            } else {
                TABLE.update(filtered, renderVersion);
            }

            /* Bind infinite scroll */
            Picasso.Helper.bindInfiniteScroll(json, TABLE, PAGE,
                function () {
                    PAGE++;

                    Picasso.Versions.fetch(Picasso.get("cwd"), true);
                }
            );

            handleOnSelection();

            if (onSuccess) onSuccess();
        };

        /* Check whether sub or root level */
        if (file) {
            /* Update cwd and store folder ID */
            Picasso.set("folderId", file.getFolderId());

            file.getFiles(updateFiles, onError, PAGE, null, true);
        } else {
            Picasso.File.getAll(updateFiles, onError, PAGE, null, true);
        }
    };

    /* Visibility - must be defined after constructor */

    if (!window.Picasso.Versions.hasOwnProperty("Visibility")) {
        window.Picasso.Versions.Visibility = {};
    }

    Picasso.Versions.Visibility.RESTORE = (1 << 0);
    Picasso.Versions.Visibility.PURGE = (1 << 1);

    /* Register view mode table callbacks */
    TABLE.setSelectionHandler(handleOnSelection);

    /* Bind handlers */
    $("#versions_home").click(handleClickCrumb);
    $("#versions_restore").click(handleClickRestore);
    $("#versions_purge").click(handleClickPurge);
    $("#versions_purge_all").click(handleClickPurge);

    Picasso.Actionbar.setFilterHandler(handleFilter);

    /* Init */
    Picasso.Versions.fetch();
})();
