/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/* Private scope */
(function () {

    /**
     * Copy data from inputs to account
     *
     * @param {Object}         inputs  Input fields ({ "id" = HTMLElement })
     * @param {Picasso.Device}  device   A #{@link Picasso.Device}
     **/

    var copyData = function (inputs, device) {
        device.setName(inputs.pica_device_name.val());
    };

    /**
     * Constructor for {@link Picasso.Device}
     *
     * @constructor
     **/

    Picasso.Device = function (data) {
        var json = Picasso.Helper.tryJSON(data);

        /* Fetch data: Load from server or restore from stringified object  */
        this._flags = Picasso.Helper.getPropOrDefValue(json, 0, "_flags");
        this._oid = Picasso.Helper.getPropOrDefValue(json, "", "id", "ID", "oid", "OID", "_oid");

        this._nick = Picasso.Helper.getPropOrDefValue(json, "", "nick", "_nick");
        this._connected = Picasso.Helper.getPropOrDefValue(json, "", "connected", "_connected");
        this._lastConnectionTime = Picasso.Helper.getPropOrDefValue(json, "", "lastConnectionTime", "_lastConnectionTime");
        this._location = Picasso.Helper.getPropOrDefValue(json, "", "location", "_location");
        this._locationLon = Picasso.Helper.getPropOrDefValue(json, "", "locationLon", "_locationLon");
        this._locationLat = Picasso.Helper.getPropOrDefValue(json, "", "locationLat", "_locationLat");

        // this._account = Picasso.Helper.getPropOrDefValue(json, "", "account", "_account");
        // this._server = Picasso.Helper.getPropOrDefValue(json, "", "server", "_server");


        // this._foldersCount = Picasso.Helper.getPropOrDefValue(json, 0, "nFolder", "_foldersCount");

        /* Create proxy to catch getter/setter calls */
        return Picasso.Helper.createProxy(this);
    };

    /**
     * Edit device via dialog
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Device.prototype.edit = function (onSuccess, onError) {
        var self = this;

        /* Show dialog */
        var dialog = new Picasso.Dialog("#pica_device_dialog",
            Picasso.Lang.get("dialog_title_edit"),
            Picasso.Lang.get("dialog_body_edit", Picasso.Lang.get("label_devices")),
            (Picasso.Dialog.Flags.VALIDATE|Picasso.Dialog.Flags.HIDE_ON_ESC));

        dialog.setAssignHandler(function (inputs) {
            inputs.pica_device_name.val(self.getNick());
            inputs.pica_device_notes.val(self.getNotes());
        });

        dialog.setOkHandler(function (inputs) {
            copyData(inputs, self);

            self.save(onSuccess, onError);
        });

        dialog.show();
    };

    /**
     * Remove device
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Device.prototype.remove = function (onSuccess, onError) {
        var url = "/wapi/devices";
        var data = {
            action: "delete",
            ID: this.getOid(),
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Save device settings
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Device.prototype.save = function (onSuccess, onError) {
        var self = this;
        var url = "/wapi/devices";
        var data = {
            action: "store",
            name: this.getNick(),
        };

        /* Add oid if any */
        if (this._oid) {
            data.oid = this._oid;
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (data, status) {
                if (data.hasOwnProperty("ID")) {
                    self.setOid(data.ID);
                }

                if (onSuccess) onSuccess(self);
            },
            onError
        );
    };

    /* Global - must be defined after constructor */

    /**
     * Get all devices
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     * @param {Number}    pageNum    Page number to load (optional)
     * @param {Number}    pageSize   Number of items per page (optional)
     **/

    Picasso.Device.getAll = function (onSuccess, onError, pageNum, pageSize) {
        var url = "/wapi/devices";
        var data = {
            action: "getAll",
            page: pageNum,
            size: pageSize
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                ary = $.map(ary, function (val) {
                    return new Picasso.Device(val);
                });

                if (onSuccess) onSuccess(ary, json);
            }, onError
        );
    };
})();
