/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/* Private scope */
(function () {

    /**
     * Constructor for {@link Picasso.File}
     *
     * @param {Object}  data  Data object
     * @constructor
     **/

    Picasso.File = function (data) {
        var json = Picasso.Helper.tryJSON(data);

        /* Fetch data: Load from server or restore from stringified object  */
        this._flags = Picasso.Helper.getPropOrDefValue(json, 0, "_flags");
        this._oid = Picasso.Helper.getPropOrDefValue(json, "", "id", "ID", "oid", "OID", "inviteID", "_oid");
        this._name = Picasso.Helper.getPropOrDefValue(json, "", "fileName", "folderName", "name", "displayName", "_name");
        this._intName = Picasso.Helper.getPropOrDefValue(json, "", "internalName", "_intName");
        this._size = Picasso.Helper.getPropOrDefValue(json, 0, "size", "_size");
        this._archiveSize = Picasso.Helper.getPropOrDefValue(json, 0, "archiveSize", "_archiveSize");
        this._resUrl = Picasso.Helper.getPropOrDefValue(json, null, "resourceURL", "_resUrl");
        this._openUrl = Picasso.Helper.getPropOrDefValue(json, null, "openURL", "_openUrl");
        this._basePath = Picasso.Helper.getPropOrDefValue(json, null, "basePath", "_basePath");
        this._path = Picasso.Helper.getPropOrDefValue(json, null, "path", "_path");

        /* Root */
        this._folderId = Picasso.Helper.getPropOrDefValue(json, null, "folderID", "_folderId");
        this._accountsCount = Picasso.Helper.getPropOrDefValue(json, 0, "membersAccount", "_accountsCount");
        this._groupsCount = Picasso.Helper.getPropOrDefValue(json, 0, "membersGroup", "_groupsCount");
        this._versions = Picasso.Helper.getPropOrDefValue(json, 0, "versions", "_versions");
        this._ownerOid = Picasso.Helper.getPropOrDefValue(json, 0, "ownerOid", "_ownerOid");
        this._isBlocked = Picasso.Helper.getPropOrDefValue(json, false, "isBlocked", "_isBlocked");

        /* File */
        this._modifiedDate = Picasso.Helper.getPropOrDefValue(json, -1, "modifiedDate", "_modifiedDate");
        this._modifiedDateMS = Picasso.Helper.getPropOrDefValue(json, -1, "modifiedDateMS", "_modifiedDateMS");
        this._modifiedOn = Picasso.Helper.getPropOrDefValue(json, "", "modifiedBy", "_modifiedOn");
        this._modifiedBy = Picasso.Helper.getPropOrDefValue(json, null, "modifiedBy", "_modifiedBy");
        this._modifiedByAccount = Picasso.Helper.getPropOrDefValue(json, null, "modifiedByAccount", "_modifiedByAccount");
        this._version = Picasso.Helper.getPropOrDefValue(json, 0, "version", "_version");
        this._publicUrl = Picasso.Helper.getPropOrDefValue(json, "", "publicURL", "_publicUrl");
        this._isStorageExceeded = Picasso.Helper.getPropOrDefValue(json, "", "isStorageExceeded", "_isStorageExceeded");

        /* Parse permissions */
        this._permission = Picasso.Permission.fromValue(
            Picasso.Helper.getPropOrDefValue(json, null,
                "permission", "invitePermission", "_permission"));
        this._publicPermission = Picasso.Permission.fromValue(
            Picasso.Helper.getPropOrDefValue(json, null,
                "publicPermission", "_publicPermission"));

        /* Update flags if unset */
        if (json && !json.hasOwnProperty("_flags")) {
            this._flags = (this._folderId ? Picasso.File.Flags.FOLDER :
                (json.hasOwnProperty("type") && "dir" === json.type ?
                    Picasso.File.Flags.DIRECTORY : Picasso.File.Flags.FILE));

            if (json.hasOwnProperty("invited") && true === json.invited) {
                this._flags |= Picasso.File.Flags.INVITED_TO;
            }

            if (json.hasOwnProperty("deleted") && true === json.deleted) {
                this._flags |= Picasso.File.Flags.DELETED;
            }

            if (json.hasOwnProperty("isFederated") && true === json.isFederated) {
                this._flags |= Picasso.File.Flags.FEDERATED;
            }
        }

        /* Parse res url */
        if (this._resUrl) {

            /* Split host from resource url */
            var match = this._resUrl.match(Picasso.File.RE_RESOURCE_URL);

            if (match) {
                var url = this._resUrl.substr(match.index + match[0].length);

                var paths = url.split("/");

                this._folderIdEnc = paths.shift();
                if (!this._path) {
                    this._path = decodeURIComponent(Picasso.Helper.joinPaths(paths));
                }
                this._url = url;
            }

            /* Reformat source URL for federated files */
            if (!!(this._flags & Picasso.File.Flags.FILE) && Picasso.get("isFederated")) {
                this._resUrl = Picasso.Helper.getPathFromURL(this._resUrl);
            }
        }

        /* Create proxy to catch getter/setter calls */
        return Picasso.Helper.createProxy(this);
    };

    /* Prototype methods */

    /**
     * Check whether file extension is an image type
     *
     * @returns {Boolean}  Either true for images types; otherwise false
     **/

    Picasso.File.prototype.isImage = function () {
        return Picasso.File.RE_IS_IMAGE.test(this._resUrl);
    };

    /**
     * Check whether file extension is a video type
     *
     * @returns {Boolean}  Either true for video types; otherwise false
     **/

    Picasso.File.prototype.isVideo = function () {
        return Picasso.File.RE_IS_VIDEO.test(this._resUrl);
    };
    /**
     * Check whether file extension is an audio type
     *
     * @returns {Boolean}  Either true for audio types; otherwise false
     **/

    Picasso.File.prototype.isAudio = function () {
        return Picasso.File.RE_IS_AUDIO.test(this._resUrl);
    };

    /**
     * Check whether file is Handled by Only Office
     *
     * @returns {Boolean}  Either true for Handled by Only Office; otherwise false
     **/

    Picasso.File.prototype.isOODocument = function () {
        return Picasso.File.OO_FILE.test(this._resUrl);
    };
    /**
     * Check whether file is txt
     *
     * @returns {Boolean}  Either true for txt file
     **/

    Picasso.File.prototype.isTXT = function () {
        return Picasso.File.TXT.test(this._resUrl);
    };

    /**
     * Check whether file is Handled by inline viewer
     *
     * @returns {Boolean}  Either true for Handled by inline viewe; otherwise false
     **/

    Picasso.File.prototype.isInline = function () {
        return Picasso.File.OO_INLINE.test(this._resUrl);
    };


    /**
     * Check whether file extension is an image type
     *
     * @returns {Boolean}  Either true for images types; otherwise false
     **/

    Picasso.File.prototype.isTifImage = function () {
        return Picasso.File.RE_IS_TIFF_IMAGE.test(this._resUrl);
    };

    /**
     * Format accounts and groups
     *
     * @returns {String} Formatted description
     **/

    Picasso.File.prototype.getDescription = function () {
        var description = "";

        if (this.is(Picasso.File.Flags.FEDERATED)) {
            description = Picasso.Lang.get("string_members_show");
        } else if (0 === this.getAccountsCount() && 0 === this.getGroupsCount()) {
            description = Picasso.Lang.get("string_members_none");
        } else if (1 === this.getAccountsCount() &&
            0 === this.getGroupsCount() &&
            Picasso.Permission.OWNER === this.getPermission() && Picasso.get("account") && Picasso.get("account").getoid() === this.getOwnerOid())
        {
            description = Picasso.Lang.get("string_members_just_you");
        } else if (0 === this.getGroupsCount()) {
            description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                "string_members_only", this.getAccountsCount());
        } else if (0 === this.getAccountsCount()) {
            description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                "string_groups_only", this.getGroupsCount());
        } else {
            description = Picasso.Lang.get(Picasso.Lang.Flags.INFLECT,
                "string_members_and_groups",
                this.getAccountsCount(), this.getGroupsCount());
        }

        return description;
    };

    /**
     * Get icon for {@link Picasso.File}
     *
     * @returns {String} File icon in a span tag
     **/

    Picasso.File.prototype.getIcon = function () {
        var glyph;

        /* Check type */
        if (this.is(Picasso.File.Flags.INVITED_TO)) {
            glyph = Picasso.Helper.createGlyph(
                "glyphicons glyphicons-send", "pica-glyph");
        } else if (this.is(Picasso.File.Flags.FOLDER)) {
            glyph = Picasso.Helper.createGlyph(
                "glyphicons glyphicons-folder-open", "pica-glyph");
        } else if (this.is(Picasso.File.Flags.DIRECTORY)) {
            glyph = Picasso.Helper.createGlyph(
                "glyphicons glyphicons-folder-closed", "pica-glyph");
        } else {
            glyph = Picasso.Helper.createGlyphForExtension(this._name);
        }

        return glyph;
    };

    /**
     * Get thumbnail url for {@link Picasso.File}
     *
     * @returns {String} Thumbnail url
     **/

    Picasso.File.prototype.getThumbnailUrl = function () {
        return Picasso.Helper.joinPaths([
            "thumb", this._url
        ]);
    };

    /**
     * Get download url for {@link Picasso.File}
     *
     * @returns {String} Download url
     **/

    Picasso.File.prototype.getDownloadUrl = function () {
        var url = this.getResUrl();

        /* Update url for folders and directories */
        if (this.is(Picasso.File.Flags.FOLDER | Picasso.File.Flags.DIRECTORY)) {
            url = Picasso.Helper.joinPaths([
                "download", this._url
            ]);
        } else {
            url = this.replaceFederatedHost(url);
        }

        return url;
    };

    /**
     * Get open url for {@link Picasso.File}
     *
     * @returns {String} Open url
     **/

    Picasso.File.prototype.getOpenUrl = function () {
        if (!this._openUrl) {
            return null;
        }

        return this.replaceFederatedHost(this._openUrl);
    };

    /**
     * Get folder id of this {@link Picasso.File} or parent folder
     *
     * @returns {String} The folder ID
     **/

    Picasso.File.prototype.getFolderId = function () {
        return (this._folderId ? this._folderId :
            Picasso.get("folderId"))
    };

    /**
     * Get folder number of maximum versions for files in this Folder {@link Picasso.File}
     *
     * @returns {String} File permission
     **/

    Picasso.File.prototype.getMaxVersions = function () {
        return (this._versions ? this._versions :
            Picasso.get("versions"))
    };

    /**
     * Get folder base Path {@link Picasso.File}
     *
     * @returns {String} Folder base Path
     **/

    Picasso.File.prototype.getBasePath = function () {
        return (this._basePath ? this._basePath :
                Picasso.get("_basePath"))
    };

    /**
     * Set folder base Path {@link Picasso.File}
     *
     * @param {String}              path  the new basePath
     * @param {Function}            onSuccess   Call on success
     * @param {Function}            onError     Call on error
     **/

    Picasso.File.prototype.setBasePath = function (path,
                                                            onSuccess, onError) {
        if (this.is(Picasso.File.Flags.FOLDER)) {
            var self = this;
            var url = Picasso.Helper.joinPaths([
                "/wapi/folders",
                this.getUrl()
            ]);

            var baseurl = window.location.protocol + "//" + window.location.host + "/";

            var data = {
                action: "setBasePath",
                basePath: path
            };

            /* Fire ajax call */
            Picasso.Helper.fireAjax(baseurl+url, data,

                    /* Success */
                    function (data, status) {
                        /* Update permission */

                        self._basePath = path;

                        if (onSuccess) {
                            onSuccess(self, status);
                        }
                    },
                    function (e, status) {
                        /* Handle folder errors */
                        switch (status) {
                            case 409:
                                Picasso.Notification.show(Picasso.Lang.get(
                                        "notification.physical_path_occupied", "warning"));
                                break;
                        }

                        if (onError) onError(e, status);
                    }

            );
        }
    };

    /**
     * Set versions this folder {@link Picasso.File}
     *
     * @param {String}              versions  the number of file versions
     * @param {Function}            onSuccess   Call on success
     * @param {Function}            onError     Call on error
     **/

    Picasso.File.prototype.setMaxVersions = function (versions,
                                                           onSuccess, onError) {
        /* Handle types */
        if (this.is(Picasso.File.Flags.FOLDER)) {
            var self = this;
            var url = Picasso.Helper.joinPaths([
                "/wapi/folders",
                this.getUrl()
            ]);

            var baseurl = window.location.protocol + "//" + window.location.host + "/";

            var data = {
                action: "setVersions",
                versions: versions
            };

            /* Fire ajax call */
            Picasso.Helper.fireAjax(baseurl+url, data,

                /* Success */
                function (data, status) {
                    /* Update permission */

                    self._versions = versions;

                    if (onSuccess) {
                        onSuccess(self, status);
                    }
                }, onError
            );
        }
    };


    /**
     * Get permission of this {@link Picasso.File} or parent folder
     *
     * @returns {Picasso.Permission} File permission
     **/

    Picasso.File.prototype.getPermission = function () {
        return (this._permission ? this._permission :
            Picasso.get("folderPermission"))
    };

    /**
     * Set public permission of this {@link Picasso.File}
     *
     * @param {Picasso.Permission}  permission  File permission
     * @param {Function}            onSuccess   Call on success
     * @param {Function}            onError     Call on error
     **/

    Picasso.File.prototype.setPublicPermission = function (permission,
                                                           onSuccess, onError) {
        /* Handle types */
        if (this.is(Picasso.File.Flags.FOLDER)) {
            var self = this;
            var url = Picasso.Helper.joinPaths([
                "/wapi/folders",
                this.getUrl()
            ]);
            var data = {
                action: "setPublicPermission",
                permission: Picasso.Permission.toString(permission)
            };

            /* Fire ajax call */
            Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (data, status) {
                    /* Update permission */
                    self._publicPermission = permission;

                    if (onSuccess) {
                        onSuccess(self, status);
                    }
                }, onError
            );
        }
    };


    /**
     * Rename given {@link Picasso.File}
     *
     * @param {String}    newName    New name for item
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.prototype.rename = function (newName, onSuccess, onError) {
        var self = this;
        var url = null;
        var data = {
            action: "rename",
            newName: newName
        };

        var oldName = this.getName();

        /* Handle types */
        if (this.is(Picasso.File.Flags.FOLDER)) {
            url = Picasso.Helper.joinPaths([
                "/wapi/folders",
                this.getUrl()
            ]);

            data.ID = this.getFolderId();
        } else {
            url = Picasso.Helper.joinPaths([
                "/wapi/files",
                Picasso.Helper.movePathUp(this.getUrl()) ///< We have to strip the name here
            ]);

            data.oldName = oldName;
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (data, status) {
                /* Complex renaming here: The name may include the
                 *                        name of the owner */
                var idx = oldName.lastIndexOf("(");

                if (-1 !== idx) {
                    self.setName(newName + " " + oldName.substr(idx));
                } else {
                    self.setName(newName);
                }

                self.setIntName(newName);
                self.setPath(newName);

                if (onSuccess) {
                    onSuccess(data, status);
                }
            }, onError
        );
    };

    /**
     * Restore given version of {@link Picasso.File}
     *
     * @param {Number}    number     Version to restore
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.prototype.restore = function (version, onSuccess, onError) {

        /* Handle types */
        if (this.is(Picasso.File.Flags.DIRECTORY|Picasso.File.Flags.FILE)) {
            var self = this;
            var url = Picasso.Helper.joinPaths([
                "/wapi/files",
                this.getUrl()
            ]);
            var data = {
                action: "restore",
                version: version
            };

            /* Fire ajax call */
            Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (data, status) {
                    if (onSuccess) {
                        onSuccess(data, status);
                    }
                }, onError
            );
        }
    };

    /**
     * Move {@link Picasso.File} and optionally copy it
     *
     * @param {Picasso.File}  target     A #{@link Picasso.File}
     * @param {String}        newName    New name for a copy (optional; copy instead of move when nonnull)
     * @param {Function}      onSuccess  Call on success
     * @param {Function}      onError    Call on error
     **/

    Picasso.File.prototype.move = function (target, newName, onSuccess, onError) {

        /* Handle types */
        if (target.is(Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY)) {
            var url = Picasso.Helper.joinPaths(["/wapi/folders", this._folderIdEnc]);
            var data = {
                method: "POST",
                action: "move",
                source: decodeURIComponent(this._path),
                target: decodeURI(Picasso.Helper.joinPaths([
                    target.getPath(),
                    (newName || this._name)
                ])),
                targetID: target.getFolderId(),
                delete: (null == newName) ///< Delete when there is no new name given
            };

            /* Fire ajax call */
            Picasso.Helper.fireAjax(url, data, onSuccess, onError);
        }
    };

    /**
     * Move {@link Picasso.File} and optionally copy it
     *
     * @param {Picasso.File}  target     A #{@link Picasso.File}
     * @param {String}        newName    New name for a copy (optional; copy instead of move when nonnull)
     * @param {Function}      onSuccess  Call on success
     * @param {Function}      onError    Call on error
     **/

    Picasso.File.prototype.purge = function (onSuccess, onError) {

        var url = Picasso.Helper.joinPaths(["/wapi/folders", this._folderIdEnc]);
        var data = {
            method: "POST",
            action: "purge"
        };
        /* Handle directories and files */
        if (!this.is(Picasso.File.Flags.FOLDER)) {
            url = Picasso.Helper.joinPaths(["/wapi/files", this._url]);
        }
        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Remove root folder, remove it and/or any sub path entirely
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.prototype.remove = function (onSuccess, onError) {
        var url = null;
        var data = {};

        /* Handle types */
        if (this.is(Picasso.File.Flags.FOLDER)) {
            url = Picasso.Helper.joinPaths([
                "/wapi/folders", this._folderIdEnc
            ]);

            data.action = "removeFolder";
            data.leave = true;
            data.removePermission = true;
            data.FolderID = encodeURIComponent(this.getFolderId());
            data.FolderName = encodeURIComponent(this.getName());
        } else {
            url = Picasso.Helper.joinPaths([
                "/wapi/files", this.getUrl()
            ]);

            data.action = "delete";
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Leave  folder
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.prototype.leave = function (onSuccess, onError) {
        var url = null;
        var data = {};

        /* Handle types */
        if (this.is(Picasso.File.Flags.FOLDER)) {
            url = Picasso.Helper.joinPaths([
                "/wapi/folders", this._folderIdEnc
            ]);

            data.action = "leave";
            data.FolderID = encodeURIComponent(this.getFolderId());
            data.FolderName = encodeURIComponent(this.getName());
            /* Fire ajax call */
            Picasso.Helper.fireAjax(url, data, onSuccess, onError);
        }


    };

    /**
     * Create subdirectory
     *
     * @param {String}    newName    Name of the new directory
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.prototype.createSubDir = function (newName, onSuccess, onError) {
        var url = Picasso.Helper.joinPaths([
            "/wapi/folders", this._folderIdEnc, this._path.replaceAll("#", "%23")
        ]);

        var data = {
            action: "createSubdir",
            dirName: newName
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };


    /**
     * Get list of members with permission
     * ({ holder: {@link Picasso.Account}, permission: {@link Picasso.Permission} })
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.prototype.getAccountsWithPermission = function (onSuccess,
                                                                 onError) {
        var url = Picasso.Helper.joinPaths(["/wapi/folders",
            this._url
        ]);
        var data = {
            action: "getAccounts"
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (json) {
                    if (onSuccess) {
                        var list = [];
                        var ary = Picasso.Helper.getResultArray(json);

                        $.each(ary, function () {
                            list.push({
                                holder: new Picasso.Account(this),
                                permission: Picasso.Permission.fromString(
                                        this.permission)
                            });
                        });

                        onSuccess(list);
                    }
                }, onError);
    };

    /**
     * Get list of groups with permission
     * ({ holder: {@link Picasso.Group}, permission: {@link Picasso.Permission} })
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.prototype.getGroupsWithPermission = function (onSuccess,
                                                               onError) {
        var url = Picasso.Helper.joinPaths(["/wapi/folders",
            this._url
        ]);
        var data = {
            action: "getGroups"
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (json) {
                    if (onSuccess) {
                        var list = [];
                        var ary = Picasso.Helper.getResultArray(json);

                        $.each(ary, function () {
                            list.push({
                                holder: new Picasso.Group(this),
                                permission: Picasso.Permission.fromString(
                                        this.permission)
                            });
                        });

                        onSuccess(list);
                    }
                }, onError);
    };

    /**
     * Compare two objects
     *
     * @param {Object}  b  A {@link Picasso.File}
     *
     * @returns {Boolean} Whether two objects are the same
     **/

    Picasso.File.prototype.equals = function (b) {
        var a = this;

        return (a && b && a._resUrl === b._resUrl);
    };

    /**
     * Get files
     *
     * @param {Function}  onSuccess    Call on success
     * @param {Function}  onError      Call on error
     * @param {Integer}   pageNum      Number of page to load (optional)
     * @param {Integer}   pageSize     Number of items per page (optional)
     * @param {Boolean}   deletedOnly  Whether to list deleted only (optional)
     * @param {String}    query        Query to filter results (optional)
     *      * @param {Boolean}    recursive        if search for files should be recursive (optional)

     **/

    Picasso.File.prototype.getFiles = function (onSuccess, onError,
                                                pageNum, pageSize,
                                                deletedOnly, query ,recursive)
    {
        /* Handle types */
        if (this.is(Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY)) {
            var url = Picasso.Helper.joinPaths([
                "/wapi/folders", this._url
            ]);
            var data = {
                action: "getFiles",
                page: pageNum,
                size: pageSize
            };

            /* Add deleted only parameter */
            if (true === deletedOnly) {
                data.deletedOnly = true;
            }

            /* Add query parameter */
            if (query) {
                data.query = query;
            }

            //check if there is already a calls on the page and cancel them
            if (Picasso.Helper.xhrPool[url] && Picasso.Helper.xhrPool[url].length > 0) {
                Picasso.Helper.abortAll(url)
                $("#files_files_table tbody .pica-spinner").hide();
            }

            if (recursive) {
                data.recursive = recursive;
            }

            /* Fire ajax call */
            Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (json) {
                    var ary = Picasso.Helper.getResultArray(json);

                    /* Create file objects */
                    ary = $.map(ary, function (val) {
                        /* Exclude some files */
                        if (false === val.inCloud || false === val.local ||
                            (true !== deletedOnly && true === val.deleted)) {
                            return null;
                        } else {
                            return new Picasso.File(val);
                        }
                    });

                    if (onSuccess) onSuccess(ary, json);
                }, onError
            );
        } else {
            if (onError) onError();
        }
    };

    /**
     * Get Link files
     *
     * @param {Function}  onSuccess    Call on success
     * @param {Function}  onError      Call on error
     * @param {Integer}   pageNum      Number of page to load (optional)
     * @param {Integer}   pageSize     Number of items per page (optional)
     * @param {String}    LinkId       Link ID
     **/

    Picasso.File.prototype.getLinkFiles = function (onSuccess, onError,
                                                pageNum, pageSize,
                                                LinkId)
    {
        /* Handle types */
        if (this.is(Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY)) {
            var url = Picasso.Helper.joinPaths([
                "/wapi/filelink", this._url
            ]);

            var baseurl = window.location.protocol + "//" + window.location.host + "/";


            var data = {
                action: "getFiles",
                page: pageNum,
                size: pageSize,
                ID:LinkId
            };

            /* Fire ajax call */
            Picasso.Helper.fireAjax(baseurl+url, data,

                    /* Success */
                    function (json) {
                        var ary = Picasso.Helper.getResultArray(json);

                        /* Create file objects */
                        ary = $.map(ary, function (val) {
                                return new Picasso.File(val);
                        });

                        if (onSuccess) onSuccess(ary, json);
                    }, onError
            );
        } else {
            if (onError) onError();
        }
    };

    /**
     * Remove link File
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.prototype.removeFileLink = function (onSuccess, onError) {

        var baseurl = window.location.protocol + "//" + window.location.host + "/";
        var url = null;
        var data = {};

        url = Picasso.Helper.joinPaths([
            "/wapi/filelink", this.getUrl()
        ]);

        data.action = "delete";

        /* Fire ajax call */
        Picasso.Helper.fireAjax(baseurl+url, data, onSuccess, onError);
    };

    /**
     * Move link {@link Picasso.File} and optionally copy it
     *
     * @param {Picasso.File}  target     A #{@link Picasso.File}
     * @param {String}        newName    New name for a copy (optional; copy instead of move when nonnull)
     * @param {Function}      onSuccess  Call on success
     * @param {Function}      onError    Call on error
     **/

    Picasso.File.prototype.moveFileLink = function (target, newName, onSuccess, onError) {

        /* Handle types */
        if (target.is(Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY)) {
            var url = Picasso.Helper.joinPaths(["/wapi/filelink", this._folderIdEnc]);
            var data = {
                method: "POST",
                action: "move",
                source: decodeURIComponent(this._path),
                target: decodeURI(Picasso.Helper.joinPaths([
                    target.getPath(),
                    (newName || this._name)
                ])),
                targetID: target.getFolderId(),
                delete: (null == newName) ///< Delete when there is no new name given
            };

            /* Fire ajax call */
            var baseurl = window.location.protocol + "//" + window.location.host + "/";

            Picasso.Helper.fireAjax(baseurl+url, data, onSuccess, onError);
        }
    };

    /**
     * Rename given {@link Picasso.File}
     *
     * @param {String}    newName    New name for item
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.prototype.renameFileLink = function (newName, onSuccess, onError) {
        var baseurl = window.location.protocol + "//" + window.location.host + "/";
        var self = this;
        var url = null;
        var data = {
            action: "rename",
            newName: newName
        };

        var oldName = this.getName();

        url = Picasso.Helper.joinPaths([
            "/wapi/filelink",
            Picasso.Helper.movePathUp(this.getUrl()) ///< We have to strip the name here
        ]);

        data.oldName = oldName;


        /* Fire ajax call */
        Picasso.Helper.fireAjax(baseurl+url, data,

                /* Success */
                function (data, status) {
                    /* Complex renaming here: The name may include the
                     *                        name of the owner */
                    var idx = oldName.lastIndexOf("(");

                    if (-1 !== idx) {
                        self.setName(newName + " " + oldName.substr(idx));
                    } else {
                        self.setName(newName);
                    }

                    self.setIntName(newName);
                    self.setPath(newName);

                    if (onSuccess) {
                        onSuccess(data, status);
                    }
                }, onError
        );
    };

    /**
     * Get file versions
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.prototype.getVersions = function (onSuccess, onError) {
        /* Handle types */
        if (this.is(Picasso.File.Flags.DIRECTORY|Picasso.File.Flags.FILE)) {
            var url = Picasso.Helper.joinPaths([
                "/wapi/files", this._folderIdEnc, this._path
            ]);
            var data = {
                action: "getVersions"
            };

            Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (json) {
                    var ary = Picasso.Helper.getResultArray(json);

                    /* Create file objects */
                    ary = $.map(ary, function (val) {
                        return new Picasso.File(val);
                    });

                    if (onSuccess) onSuccess(ary, json);
                }, onError
            );
        } else {
            if (onError) onError();
        }
    };

    /**
     * Check whether a file exists in {@link Picasso.File}
     *
     * @param {String}    name       File name
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.prototype.exists = function (name, onSuccess, onError) {
        /* Handle types */
        if (this.is(Picasso.File.Flags.FOLDER|Picasso.File.Flags.DIRECTORY)) {
            var url = Picasso.Helper.joinPaths([
                "/wapi/files", this._folderIdEnc, this._path, name
            ]);
            var data = {
                action: "exists"
            };

            Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (json, status) {
                    if (onSuccess) {
                        onSuccess((200 === status));
                    }
                },

                /* Error */
                function (e, status) {
                    /* This is a bit weird, but 404 means file not found */
                    if (404 === status) {
                        if (onSuccess) {
                            onSuccess(false);
                        }
                    } else if (onError) {
                        onError(e, status);
                    }
                }
            );
        } else {
            if (onError) onError();
        }
    };

    /**
     * Convert object to string
     *
     * @returns {String} String representation of object
     **/

    Picasso.File.prototype.toString = function () {
        return this._name;
    };

    /* Global - must be defined after constructor */

    /**
     * Create root folder
     *
     * @param {String}    newName    Name of the new directory
     * @param {Boolean}   encrypted  Either this is an encrypted folder or not.
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.File.createRootDir = function (newName, encrypted, onSuccess, onError) {
        var url = "/wapi/folders";
        var data = {
            name: newName,
            action: "create",
            encrypted: encrypted
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Replace the federated host with the local host for proxying.
     *
     * @param url
     * @returns {string | void}
     */

    Picasso.File.prototype.replaceFederatedHost = function (url) {

        var federatedServices = Picasso.get("fedServices");

        if (!federatedServices) {
            return url;
        }

        for (var i = 0; i <= federatedServices.length; i++) {
            if (url.startsWith(federatedServices[i])) {
                return url.replace(federatedServices[i], window.location.origin);
            }
        }

        return url;
    };

    /**
     * Get all folder/file links for current user account
     *
     * @param {Function}  onSuccess    Call on success
     * @param {Function}  onError      Call on error
     * @param {Integer}   pageNum      Page number to load (optional)
     * @param {Integer}   pageSize     Number of items per page (optional)
     * @param {Boolean}   deletedOnly  Whether to list deleted only (optional)
     * @param {String}    query        Query to filter results (optional)
     * @param {String}    filter       filter flag to fetch owned folders or shared  (optional)
     **/

    Picasso.File.getAll = function (onSuccess, onError, pageNum, pageSize, deletedOnly, query, filter)
    {
        var url = "/wapi/folders";
        var data = {
            action: "getAll",
            page: pageNum,
            size: pageSize
        };

        /* Add deleted only parameter */
        if (true === deletedOnly) {
            data.deletedOnly = true;
        }

        /* Add query parameter */
        if (query) {
            data.query = query;
        }
        /* Add query parameter */
        if (filter) {
            data.filter = filter;
        }

        //check if there is already a calls on the page and cancel them
        if (Picasso.Helper.xhrPool[url] && Picasso.Helper.xhrPool[url].length > 0) {
            Picasso.Helper.abortAll(url)
            $("#files_files_table tbody .pica-spinner").hide();
        }

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                /* Create file objects */
                ary = $.map(ary, function (val) {
                    /* Exclude some files */
                    if (false === val.inCloud || false === val.local ||
                        (true === deletedOnly && true === val.invited) ||
                        true === val.deleted)
                    {
                        return null;
                    } else {
                        return new Picasso.File(val);
                    }
                });

                if (onSuccess) onSuccess(ary, json);
            }, onError
        );
    };

    /**
     * Get folder by filelink for current user account
     *
     * @param {Function}  onSuccess    Call on success
     * @param {Function}  onError      Call on error
     * @param {Integer}   pageNum      Page number to load (optional)
     * @param {Integer}   pageSize     Number of items per page (optional)
     * @param {String}    linkId       File Link ID
     **/

    Picasso.File.getLinkFolder = function (onSuccess, onError, pageNum, pageSize, linkId)
    {
        var url = "/wapi/filelink";
        var data = {
            action: "getFileInfo",
            page: pageNum,
            size: pageSize,
            ID:linkId
        };



        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

                /* Success */
                function (json) {
                    var ary = Picasso.Helper.getResultArray(json);

                    /* Create file objects */
                    ary = $.map(ary, function (val) {
                            return new Picasso.File(val);
                    });

                    if (onSuccess) onSuccess(ary, json);
                }, onError
        );
    };

    Picasso.File.getSingleFolder = function(onSuccess, onError, folderIdEnc) {
        var url = Picasso.Helper.joinPaths([
            "/wapi/folders", folderIdEnc
                ]);
        var data = {
            action: "getInfo"
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                var folder = new Picasso.File(ary[0]);

                if (onSuccess) onSuccess(folder);
            }, onError
        );
    };

    /* Flags - must be defined after constructor */

    if (!window.Picasso.File.hasOwnProperty("Flags")) {
        window.Picasso.File.Flags = {};
    }

    /* Flags */
    Picasso.File.Flags.FOLDER = (1 << 0);
    Picasso.File.Flags.DIRECTORY = (1 << 1);
    Picasso.File.Flags.FILE = (1 << 2);
    Picasso.File.Flags.DELETED = (1 << 3);
    Picasso.File.Flags.INVITED_TO = (1 << 4);
    Picasso.File.Flags.FEDERATED = (1 << 5);

    /* Globals */
    Picasso.File.RE_RESOURCE_URL = new RegExp("\/(?:wapi|filestable|files|download|getlink)+\/");
    Picasso.File.RE_IS_IMAGE = new RegExp("\.(jpeg|jpg|gif|png|tif|tiff)$", "i");

    Picasso.File.RE_IS_VIDEO = new RegExp("\.(mp4|flv|rtmp|rtmpv|m4v|ogv|webmv|m3uv|m3u8v|mov|webm)$", "i");
    Picasso.File.RE_IS_AUDIO = new RegExp("\.(wav|mp3|ogg|oga|webma|fla|flac|m3u8a|rtmpa|djvu)$", "i");
    Picasso.File.RE_IS_TIFF_IMAGE = new RegExp("\.(tif|tiff)$", "i");
    Picasso.File.OO_FILE = new  RegExp("\.(docx|txt|xlsx|csv|pptx|rtf)$", "i");
    Picasso.File.OO_INLINE = new  RegExp("\.(pdf|txt)$", "i");
    Picasso.File.TXT = new  RegExp("\.(txt)$", "i");
})();
