/* jshint strict: true, trailing: true, loopfunc: true, browser: true, jquery: true, devel: true, maxerr: 500 */
"use strict";

if (!window.hasOwnProperty("Picasso")) {
    window.Picasso = {};
}

/* Private scope */
(function () {

    /**
     * Constructor for {@link Picasso.Server}
     *
     * @constructor
     **/

    Picasso.Server = function (data) {
        var json = Picasso.Helper.tryJSON(data);

        /* Fetch data: Load from server or restore from stringified object  */
        this._flags = Picasso.Helper.getPropOrDefValue(json, 0, "_flags");
        this._oid = Picasso.Helper.getPropOrDefValue(json, "", "nodeID", "id", "ID", "oid", "OID", "_oid");
        this._name = Picasso.Helper.getPropOrDefValue(json, "", "name", "displayName", "_name");

        this._lastUsageTime = Picasso.Helper.getPropOrDefValue(json, "", "lastUsageTime", "_lastUsageTime");
        this._lastOnlineTime = Picasso.Helper.getPropOrDefValue(json, "", "lastOnlineTime", "_lastOnlineTime");

        this._programVersion = Picasso.Helper.getPropOrDefValue(json, "", "programVersion", "_programVersion");

        this._memoryUsage = Picasso.Helper.getPropOrDefValue(json, "", "memoryUsage", "_memoryUsage");
        this._memoryUsageBytes = Picasso.Helper.getPropOrDefValue(json, "", "memoryUsageBytes", "_memoryUsageBytes");
        this._cpuUsage = Picasso.Helper.getPropOrDefValue(json, "", "cpuUsage", "_cpuUsage");

        this._spaceUsageBytes = Picasso.Helper.getPropOrDefValue(json, "", "spaceUsageBytes", "_spaceUsageBytes");
        this._spaceFreeBytes = Picasso.Helper.getPropOrDefValue(json, "", "spaceFreeBytes", "_spaceFreeBytes");

        this._devicesConnectedCount = Picasso.Helper.getPropOrDefValue(json, "", "nDevicesConnected", "_devicesConnected");
        this._devicesCount = Picasso.Helper.getPropOrDefValue(json, "", "nDevices", "_devices");
        this._activeSessionsCount = Picasso.Helper.getPropOrDefValue(json, "", "nActiveSessions", "_sessionsActiveCount");
        this._sessionsCount = Picasso.Helper.getPropOrDefValue(json, "", "nSessions", "_sessionsCount");
        this._accountsCount = Picasso.Helper.getPropOrDefValue(json, "", "nAccounts", "_accountsCount");
        this._foldersCount = Picasso.Helper.getPropOrDefValue(json, "", "nFolders", "_foldersCount");

        this._webUrl = Picasso.Helper.getPropOrDefValue(json, "", "webURL", "_webUrl");
        this._validationDate = Picasso.Helper.getPropOrDefValue(json, "", "validationDate", "_validationDate");
        this._status = Picasso.Helper.getPropOrDefValue(json, "", "status", "_status");

        /* Create proxy to catch getter/setter calls */
        return Picasso.Helper.createProxy(this);
    };

    /**
     * Get all servers
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     **/

    Picasso.Server.getAll = function (onSuccess, onError) {
        var url = "/wapi/servers";
        var data = {
            action: "getAll"
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                ary = $.map(ary, function (val) {
                    return new Picasso.Server(val);
                });

                if (onSuccess) {
                    onSuccess(ary, json);
                }
            }, onError
        );
    };

    /**
     * Remove {@link Picasso.Server}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Server.prototype.removeNode = function (onSuccess, onError) {
        var url = "/wapi/servers";
        var data = {
            action: "delete",
            targetServerID: this.getOid()
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Restart {@link Picasso.Server}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Server.prototype.restartNode = function (onSuccess, onError) {
        var url = "/wapi/servers";
        var data = {
            action: "restart",
            targetServerID: this.getOid()
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Create {@link Picasso.Server (federated service)}
     *
     * @param serviceURL URL of the service
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Server.createFederatedService = function (serviceURL, onSuccess, onError) {
        var url = "/wapi/federation";
        var data = {
            action: "create",
            serviceURL: serviceURL
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Get all {@link Picasso.Server (federated service)}
     *
     * @param onSuccess
     * @param onError
     */

    Picasso.Server.getAllFederatedServices = function (onSuccess, onError) {
        var url = "/wapi/federation";
        var data = {
            action: "getAll"
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data,

            /* Success */
            function (json) {
                var ary = Picasso.Helper.getResultArray(json);

                ary = $.map(ary, function (val) {
                    return new Picasso.Server(val);
                });

                if (onSuccess) onSuccess(ary, json);
            }, onError
        );
    };

    /**
     * Remove {@link Picasso.Server (federated Service)}
     *
     * @param {Function}  onSuccess  Call on success
     * @param {Function}  onError    Call on error
     *
     * @returns {Promise} Returns the promise object
     **/

    Picasso.Server.prototype.removeFederatedService = function (onSuccess, onError) {
        var url = "/wapi/federation";
        var data = {
            action: "delete",
            serviceURL: this.getOid()
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };

    /**
     * Validate {@link Picasso.Server (federated Service)}
     *
     * @param onSuccess
     * @param onError
     */

    Picasso.Server.prototype.validateFederatedService = function (onSuccess, onError) {
        var url = "/wapi/federation";
        var data = {
            action: "confirm",
            serviceURL: this.getOid()
        };

        /* Fire ajax call */
        Picasso.Helper.fireAjax(url, data, onSuccess, onError);
    };
})();